/*
 *  linux/arch/arm/mach-uniphier/lpctl.c
 *
 *  Copyright (C) 2012 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/init.h>
#include <mach/lpu-regs.h>
#include <mach/lpd-regs.h>
#include <mach/lps-regs.h>
#include <mach/backup-ram-layout.h>

#if defined(CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
#define LPD_D_SAVE_SIZE			9 * 4
#define LPD_D_NUM				4
#define LPD_SAVEDATA_SIZE		LPD_D_SAVE_SIZE * LPD_D_NUM

#define LPD_D0_OFFSET			LPD_D_SAVE_SIZE * 0
#define LPD_D1_OFFSET			LPD_D_SAVE_SIZE * 1
#define LPD_D2_OFFSET			LPD_D_SAVE_SIZE * 2
#define LPD_D3_OFFSET			LPD_D_SAVE_SIZE * 3

typedef struct {
	unsigned long db0;
	unsigned long db1;
	unsigned long db2;
	unsigned long db3;
	unsigned long db4;
	unsigned long dl1;
	unsigned long dl2;
	unsigned long dm;
	unsigned long dg;
} lpctl_lpd_data_t;
#endif

typedef struct {
	unsigned long paddr;	/* physical address */
	unsigned long size;		/* map size */
	unsigned long vaddr;	/* virtual address */
} lpctl_map_t;


lpctl_map_t lpctl_map[] = {
	{ .paddr = LPD_BASE,				.size = 0x400000, },
	{ .paddr = LPS_BASE,				.size = 0x10000,  },
	{ .paddr = LPU_BASE,				.size = 0x2410000,},
};

#define LPCTL_MAP_NR (sizeof(lpctl_map) / sizeof(lpctl_map[0]))

static void lpctl_destroy_iomap(void)
{
	int i;

	for (i = 0; i < LPCTL_MAP_NR; i++) {
		iounmap((void *)(lpctl_map[i].vaddr));
	}
}

static int lpctl_create_iomap(void)
{
	unsigned long vaddr;
	int i;

	for (i = 0; i < LPCTL_MAP_NR; i++) {
		vaddr = (unsigned long)ioremap(
				lpctl_map[i].paddr, lpctl_map[i].size);
		if (!vaddr) {
			goto lpctl_create_iomap_error;
		} else {
			lpctl_map[i].vaddr = vaddr;
		}
	}
	return 0;

lpctl_create_iomap_error:
	lpctl_destroy_iomap();
	return -ENOMEM;
}

static unsigned long lpctl_read_reg(unsigned long addr)
{
	int i;
	unsigned long vaddr;

	for(i=0; i<LPCTL_MAP_NR; i++) {
		if((addr >= lpctl_map[i].paddr) && (addr < (lpctl_map[i].paddr + lpctl_map[i].size))) {
			vaddr = (addr - lpctl_map[i].paddr) + lpctl_map[i].vaddr;
			break;
		}
	}
	if(i >= LPCTL_MAP_NR) {
		BUG();
	}

	return *(volatile unsigned long *)vaddr;
}

static void lpctl_write_reg(unsigned long addr, unsigned long data)
{
	int i;
	unsigned long vaddr;

	for(i=0; i<LPCTL_MAP_NR; i++) {
		if((addr >= lpctl_map[i].paddr) && (addr < (lpctl_map[i].paddr + lpctl_map[i].size))) {
			vaddr = (addr - lpctl_map[i].paddr) + lpctl_map[i].vaddr;
			break;
		}
	}
	if(i >= LPCTL_MAP_NR) {
		BUG();
	}

	*(volatile unsigned long *)vaddr = data;
}


/*
 * Setting LPU before lowpower
 */
void lpctl_lowpower_init(void)
{
#ifdef CONFIG_MACH_MN2WS0220_REF
	lpctl_write_reg(LPUCENDC(0),0x0000008f);
	lpctl_write_reg(LPUCENDC(1),0x0000008f);
	lpctl_write_reg(LPUCENDC(2),0x0000008f);
	
	// set parameter
	lpctl_write_reg(LPUSCTLB(0),0x00ff0008);
	lpctl_write_reg(LPUSSETC(0),0x00100000);
	lpctl_write_reg(LPUACTLD(0),0x00010300);
	lpctl_write_reg(LPUACTLE(0),0x000a03FF);
	lpctl_write_reg(LPUSCTLB(1),0x00ff0008);
	lpctl_write_reg(LPUSSETC(1),0x00100000);
	lpctl_write_reg(LPUACTLD(1),0x00010300);
	lpctl_write_reg(LPUACTLE(1),0x000a03FF);
	lpctl_write_reg(LPUSCTLB(2),0x00ff0008);
	lpctl_write_reg(LPUSSETC(2),0x00100000);
	lpctl_write_reg(LPUACTLD(2),0x00010300);
	lpctl_write_reg(LPUACTLE(2),0x000a03FF);

#elif defined (CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
	while(lpctl_read_reg(LPUSSTAC(0)) & (unsigned long)0x00000001);
	while(lpctl_read_reg(LPUSSTAC(1)) & (unsigned long)0x00000001);

	// set parameter
	lpctl_write_reg(LPULPCTLA(0),(unsigned long)0x0cff0000);
	lpctl_write_reg(LPULPCTLA(1),(unsigned long)0x0cff0000);
	lpctl_write_reg(LPUSSETB(0),(unsigned long)0x200a0a01);
	lpctl_write_reg(LPUSSETB(1),(unsigned long)0x200a0a01);
#endif
}

/*
 * Stop LPU IF
 */
void lpctl_if_stop(void)
{
	unsigned long usecs = 10;
	
#ifdef CONFIG_MACH_MN2WS0220_REF
//	lpctl_write_reg(LPU_MOPTION_A(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_B(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_C(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_D(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_E(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_F(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_G(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_H(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_I(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_J(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_L(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_M(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_O(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_P(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_Q(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_R(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_S(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_T(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_U(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_V(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_W(0),0x00000000);
	
//	lpctl_write_reg(LPU_MOPTION_A(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_B(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_C(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_D(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_E(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_F(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_G(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_H(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_I(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_J(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_L(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_M(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_O(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_P(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_Q(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_R(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_S(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_T(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_U(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_V(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_W(1),0x00000000);
	
//	lpctl_write_reg(LPU_MOPTION_A(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_B(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_C(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_D(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_E(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_F(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_G(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_H(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_I(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_J(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_L(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_M(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_O(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_P(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_Q(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_R(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_S(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_T(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_U(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_V(2),0x00000000);
	lpctl_write_reg(LPU_MOPTION_W(2),0x00000000);
	
#elif defined (CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
//	lpctl_write_reg(LPU_MOPTION_A(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_B(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_C(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_D(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_E(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_F(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_G(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_H(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_I(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_J(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_K(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_L(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_M(0),0x00000000);
	lpctl_write_reg(LPU_MOPTION_N(0),0x00000000);
	
//	lpctl_write_reg(LPU_MOPTION_A(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_B(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_C(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_D(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_E(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_F(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_G(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_H(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_I(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_J(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_K(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_L(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_M(1),0x00000000);
	lpctl_write_reg(LPU_MOPTION_N(1),0x00000000);
#endif
	
	lpctl_write_reg(LPS_RSTCTL,0x00600030);
	
	/* 10us Wait */
	udelay(usecs);
}

/*
 * Save LPD register
 */
void lpctl_save_data(void)
{
	unsigned long *buf;
#if defined(CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)
	lpctl_lpd_data_t *d0, *d1, *d2, *d3;
#endif
	
	void __iomem *uniphier_get_backup_ram_base(void); 
	buf = (unsigned long *)((unsigned long)uniphier_get_backup_ram_base() + LPD_SAVEDATA_OFFSET);

#ifdef CONFIG_MACH_MN2WS0220_REF
	*buf++ = lpctl_read_reg(LPD_DEN_A(0, 0, 0));
	*buf++ = lpctl_read_reg(LPD_DEN_A(0, 0, 1));
	*buf++ = lpctl_read_reg(LPD_DEN_A(0, 1, 0));
	*buf++ = lpctl_read_reg(LPD_DEN_A(0, 1, 1));
	*buf++ = lpctl_read_reg(LPD_DEN_A(1, 0, 0));
	*buf++ = lpctl_read_reg(LPD_DEN_A(1, 0, 1));
	*buf++ = lpctl_read_reg(LPD_DEN_A(1, 1, 0));
	*buf++ = lpctl_read_reg(LPD_DEN_A(1, 1, 1));

	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,0));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,1));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,2));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,3));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,4));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,5));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,6));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,1,7));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,0));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,1));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,2));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,3));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,4));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,5));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,6));
	*buf++ = lpctl_read_reg(LPD_WTI(0,0,0,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,1,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,1,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,0,0,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,0,0,7));

	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,0));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,1));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,2));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,3));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,4));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,5));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,6));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,1,7));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,0));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,1));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,2));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,3));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,4));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,5));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,6));
	*buf++ = lpctl_read_reg(LPD_WTI(0,1,0,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,1,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,1,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(0,1,0,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(0,1,0,7));
	
	*buf++ = lpctl_read_reg(LPD_BTI(0,0,1));
	*buf++ = lpctl_read_reg(LPD_BTI(0,0,0));
	*buf++ = lpctl_read_reg(LPD_WTI_D(0,0,1));
	*buf++ = lpctl_read_reg(LPD_WTI_D(0,0,0));
	*buf++ = lpctl_read_reg(LPD_CTLTI(0,0,1));
	*buf++ = lpctl_read_reg(LPD_CTLTI(0,0,0));

	*buf++ = lpctl_read_reg(LPD_BTI(0,1,1));
	*buf++ = lpctl_read_reg(LPD_BTI(0,1,0));
	*buf++ = lpctl_read_reg(LPD_WTI_D(0,1,1));
	*buf++ = lpctl_read_reg(LPD_WTI_D(0,1,0));
	*buf++ = lpctl_read_reg(LPD_CTLTI(0,1,1));
	*buf++ = lpctl_read_reg(LPD_CTLTI(0,1,0));


	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,0));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,1));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,2));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,3));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,4));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,5));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,6));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,1,7));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,0));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,1));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,2));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,3));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,4));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,5));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,6));
	*buf++ = lpctl_read_reg(LPD_WTI(1,0,0,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,1,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,1,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,0,0,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,0,0,7));

	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,0));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,1));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,2));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,3));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,4));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,5));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,6));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,1,7));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,0));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,1));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,2));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,3));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,4));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,5));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,6));
	*buf++ = lpctl_read_reg(LPD_WTI(1,1,0,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,1,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,1,7));

	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,0));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,1));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,2));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,3));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,4));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,5));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,6));
	*buf++ = lpctl_read_reg(LPD_RTI_C(1,1,0,7));
	*buf++ = lpctl_read_reg(LPD_RTI_D(1,1,0,7));
	
	*buf++ = lpctl_read_reg(LPD_BTI(1,0,1));
	*buf++ = lpctl_read_reg(LPD_BTI(1,0,0));
	*buf++ = lpctl_read_reg(LPD_WTI_D(1,0,1));
	*buf++ = lpctl_read_reg(LPD_WTI_D(1,0,0));
	*buf++ = lpctl_read_reg(LPD_CTLTI(1,0,1));
	*buf++ = lpctl_read_reg(LPD_CTLTI(1,0,0));

	*buf++ = lpctl_read_reg(LPD_BTI(1,1,1));
	*buf++ = lpctl_read_reg(LPD_BTI(1,1,0));
	*buf++ = lpctl_read_reg(LPD_WTI_D(1,1,1));
	*buf++ = lpctl_read_reg(LPD_WTI_D(1,1,0));
	*buf++ = lpctl_read_reg(LPD_CTLTI(1,1,1));
	*buf++ = lpctl_read_reg(LPD_CTLTI(1,1,0));
	
#elif defined (CONFIG_MACH_MN2WS0251_REF) || defined(CONFIG_MACH_MN2WS0270_REF)	
	d0 = (lpctl_lpd_data_t *)((unsigned long)buf + LPD_D0_OFFSET);
	d1 = (lpctl_lpd_data_t *)((unsigned long)buf + LPD_D1_OFFSET);
	d2 = (lpctl_lpd_data_t *)((unsigned long)buf + LPD_D2_OFFSET);
	d3 = (lpctl_lpd_data_t *)((unsigned long)buf + LPD_D3_OFFSET);

	d0->db0	= lpctl_read_reg(LPD_DB(0, 0, 0, 0));
	d0->db1	= lpctl_read_reg(LPD_DB(0, 0, 0, 1));
	d0->db2	= lpctl_read_reg(LPD_DB(0, 0, 0, 2));
	d0->db3	= lpctl_read_reg(LPD_DB(0, 0, 0, 3));
	d0->db4	= lpctl_read_reg(LPD_DB(0, 0, 0, 4));
	d0->dl1	= lpctl_read_reg(LPD_DL(0, 0, 0, 1));
	d0->dl2	= lpctl_read_reg(LPD_DL(0, 0, 0, 2));
	d0->dm	= lpctl_read_reg(LPD_DM(0, 0, 0));
	d0->dg	= lpctl_read_reg(LPD_DG(0, 0, 0));

	d1->db0	= lpctl_read_reg(LPD_DB(0, 0, 1, 0));
	d1->db1	= lpctl_read_reg(LPD_DB(0, 0, 1, 1));
	d1->db2	= lpctl_read_reg(LPD_DB(0, 0, 1, 2));
	d1->db3	= lpctl_read_reg(LPD_DB(0, 0, 1, 3));
	d1->db4	= lpctl_read_reg(LPD_DB(0, 0, 1, 4));
	d1->dl1	= lpctl_read_reg(LPD_DL(0, 0, 1, 1));
	d1->dl2	= lpctl_read_reg(LPD_DL(0, 0, 1, 2));
	d1->dm	= lpctl_read_reg(LPD_DM(0, 0, 1));
	d1->dg	= lpctl_read_reg(LPD_DG(0, 0, 1));

	d2->db0	= lpctl_read_reg(LPD_DB(1, 1, 0, 0));
	d2->db1	= lpctl_read_reg(LPD_DB(1, 1, 0, 1));
	d2->db2	= lpctl_read_reg(LPD_DB(1, 1, 0, 2));
	d2->db3	= lpctl_read_reg(LPD_DB(1, 1, 0, 3));
	d2->db4	= lpctl_read_reg(LPD_DB(1, 1, 0, 4));
	d2->dl1	= lpctl_read_reg(LPD_DL(1, 1, 0, 1));
	d2->dl2	= lpctl_read_reg(LPD_DL(1, 1, 0, 2));
	d2->dm	= lpctl_read_reg(LPD_DM(1, 1, 0));
	d2->dg	= lpctl_read_reg(LPD_DG(1, 1, 0));

	d3->db0	= lpctl_read_reg(LPD_DB(1, 1, 1, 0));
	d3->db1	= lpctl_read_reg(LPD_DB(1, 1, 1, 1));
	d3->db2	= lpctl_read_reg(LPD_DB(1, 1, 1, 2));
	d3->db3	= lpctl_read_reg(LPD_DB(1, 1, 1, 3));
	d3->db4	= lpctl_read_reg(LPD_DB(1, 1, 1, 4));
	d3->dl1	= lpctl_read_reg(LPD_DL(1, 1, 1, 1));
	d3->dl2	= lpctl_read_reg(LPD_DL(1, 1, 1, 2));
	d3->dm	= lpctl_read_reg(LPD_DM(1, 1, 1));
	d3->dg	= lpctl_read_reg(LPD_DG(1, 1, 1));
#endif
}

static int __init lpctl_init( void )
{
	return lpctl_create_iomap();
}

early_initcall(lpctl_init);
